/*
 * FixScript v0.4 - http://www.fixscript.org/
 * Copyright (c) 2018-2020 Martin Dvorak <jezek2@advel.cz>
 *
 * This software is provided 'as-is', without any express or implied warranty.
 * In no event will the authors be held liable for any damages arising from
 * the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose, 
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 * 1. The origin of this software must not be misrepresented; you must not
 *    claim that you wrote the original software. If you use this software
 *    in a product, an acknowledgment in the product documentation would be
 *    appreciated but is not required.
 * 2. Altered source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 3. This notice may not be removed or altered from any source distribution.
 */

#include <stdio.h>
#include <stdlib.h>
#include "fixscript.h"

typedef struct {
   Value value;
} NativeRef;


static void *native_ref_handle_func(Heap *heap, int op, void *p1, void *p2)
{
   NativeRef *ref = p1, *copy;

   switch (op) {
      case HANDLE_OP_FREE:
         free(ref);
         break;

      case HANDLE_OP_COPY:
         copy = calloc(1, sizeof(NativeRef));
         if (!copy) return NULL;
         copy->value = ref->value;
         return copy;

      case HANDLE_OP_MARK_REFS:
         fixscript_mark_ref(heap, ref->value);
         break;
         
      case HANDLE_OP_COPY_REFS:
         ref->value = fixscript_copy_ref(p2, ref->value);
         break;
   }
   return NULL;
}


static Value get_func_name_func(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   char *script_name = NULL;
   char *func_name = NULL;
   int func_num_params;
   int err;
   Value ret = fixscript_int(0);
   Value arr, val;
   
   err = fixscript_get_function_name(heap, params[0], &script_name, &func_name, &func_num_params);
   if (err) {
      ret = fixscript_error(heap, error, err);
      goto error;
   }

   arr = fixscript_create_array(heap, 0);
   if (!arr.value) {
      ret = fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
      goto error;
   }

   val = fixscript_create_string(heap, script_name, -1);
   if (!val.value) {
      ret = fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
      goto error;
   }

   err = fixscript_append_array_elem(heap, arr, val);
   if (err) {
      ret = fixscript_error(heap, error, err);
      goto error;
   }

   val = fixscript_create_string(heap, func_name, -1);
   if (!val.value) {
      ret = fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
      goto error;
   }

   err = fixscript_append_array_elem(heap, arr, val);
   if (err) {
      ret = fixscript_error(heap, error, err);
      goto error;
   }

   err = fixscript_append_array_elem(heap, arr, fixscript_int(func_num_params));
   if (err) {
      ret = fixscript_error(heap, error, err);
      goto error;
   }

   ret = arr;

error:
   free(script_name);
   free(func_name);
   return ret;
}


static Value create_handle_func(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   return fixscript_create_handle(heap, 0, (void *)1, NULL);
}


static Value overrided_native_func(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   return fixscript_int(123);
}


static Value test_alt_heap(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   Heap *alt_heap = data;
   Value val;
   static Value alt_val;
   int err;

   if (!alt_val.value) {
      alt_val = fixscript_create_array(alt_heap, 2);
      if (!alt_val.value) {
         return fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
      }
      fixscript_ref(alt_heap, alt_val);
   }

   err = fixscript_clone_between(alt_heap, heap, params[0], &val, NULL, NULL, NULL);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   err = fixscript_set_array_elem(alt_heap, alt_val, params[1].value, val);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   fixscript_collect_heap(alt_heap);
   return fixscript_int(val.value);
}


static Value test_alt_heap_func_ref0(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   Heap *alt_heap = data;
   Value val;
   int err;

   err = fixscript_clone_between(alt_heap, heap, params[0], &val, NULL, NULL, NULL);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   err = fixscript_clone_between(heap, alt_heap, val, &val, NULL, NULL, NULL);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   return val;
}


static Value test_alt_heap_func_ref1(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   Heap *alt_heap = data;
   Value val;
   char *s, *error_msg;
   int err;

   err = fixscript_clone_between(alt_heap, heap, params[0], &val, NULL, NULL, NULL);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   err = fixscript_clone_between(heap, alt_heap, val, &val, (LoadScriptFunc)fixscript_load_file, ".", &error_msg);
   if (err) {
      if (error_msg) {
         *error = fixscript_create_error_string(heap, error_msg);
         free(error_msg);
         return fixscript_int(0);
      }
      return fixscript_error(heap, error, err);
   }

   err = fixscript_to_string(heap, val, 0, &s, NULL);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   val = fixscript_create_string(heap, s, -1);
   free(s);
   if (!val.value) {
      return fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
   }

   return val;
}


static Value test_alt_heap_func_ref2(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   Heap *alt_heap = data;
   Value val;
   char *s, *error_msg = NULL;
   int err;

   if (params[1].value) {
      err = fixscript_clone_between(alt_heap, heap, params[0], &val, (LoadScriptFunc)fixscript_load_file, ".", &error_msg);
   }
   else {
      err = fixscript_clone_between(alt_heap, heap, params[0], &val, NULL, NULL, NULL);
   }
   if (err) {
      if (error_msg) {
         *error = fixscript_create_error_string(heap, error_msg);
         free(error_msg);
         return fixscript_int(0);
      }
      return fixscript_error(heap, error, err);
   }

   err = fixscript_to_string(alt_heap, val, 0, &s, NULL);
   if (err) {
      return fixscript_error(heap, error, err);
   }

   val = fixscript_create_string(heap, s, -1);
   free(s);
   if (!val.value) {
      return fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
   }

   return val;
}


static Value dummy_func(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   *error = fixscript_create_error_string(heap, "dummy function");
   return fixscript_int(0);
}


static Value create_native_ref(Heap *heap, Value *error, int num_params, Value *params, void *data)
{
   NativeRef *ref;
   Value ret;

   ref = calloc(1, sizeof(NativeRef));
   if (!ref) {
      return fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
   }
   ref->value = params[0];

   ret = fixscript_create_value_handle(heap, 1, ref, native_ref_handle_func);
   if (!ret.value) {
      return fixscript_error(heap, error, FIXSCRIPT_ERR_OUT_OF_MEMORY);
   }
   return ret;
}


int main(int argc, char **argv)
{
   Heap *heap, *alt_heap;
   Value val, error;
   char *error_msg;

   heap = fixscript_create_heap();
   fixscript_register_native_func(heap, "get_func_name#1", get_func_name_func, NULL);
   fixscript_register_native_func(heap, "create_handle#0", create_handle_func, NULL);
   fixscript_register_native_func(heap, "overrided_native_func1#0", overrided_native_func, NULL);
   fixscript_register_native_func(heap, "overrided_native_func2#0", overrided_native_func, NULL);
   fixscript_register_native_func(heap, "create_native_ref#1", create_native_ref, NULL);

   alt_heap = fixscript_create_heap();
   fixscript_register_native_func(heap, "test_alt_heap#2", test_alt_heap, alt_heap);
   fixscript_register_native_func(heap, "test_alt_heap_func_ref0#1", test_alt_heap_func_ref0, alt_heap);
   fixscript_register_native_func(heap, "test_alt_heap_func_ref1#1", test_alt_heap_func_ref1, alt_heap);
   fixscript_register_native_func(heap, "test_alt_heap_func_ref2#2", test_alt_heap_func_ref2, alt_heap);
   fixscript_register_native_func(heap, "dummy_native_func#0", dummy_func, NULL);

   fixscript_register_native_func(alt_heap, "get_func_name#1", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "create_handle#0", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "overrided_native_func1#0", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "overrided_native_func2#0", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "test_alt_heap#2", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "test_alt_heap_func_ref0#1", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "test_alt_heap_func_ref1#1", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "test_alt_heap_func_ref2#2", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "dummy_native_func#0", dummy_func, NULL);
   fixscript_register_native_func(alt_heap, "create_native_ref#1", dummy_func, NULL);

   Script *script = fixscript_load_file(heap, "test", &error_msg, ".");
   if (!script) {
      printf("%s\n", error_msg);
      return 0;
   }
   val = fixscript_run(heap, script, "test#3", &error, fixscript_int(1), fixscript_int(2), fixscript_int(3));
   fixscript_dump_value(heap, val, 1);
   fixscript_dump_value(heap, error, 1);
   printf("calling function dynamically:\n");
   val = fixscript_call(heap, val, 1, &error, fixscript_create_string(heap, "hello!", -1));
   fixscript_dump_value(heap, val, 1);
   fixscript_dump_value(heap, error, 1);
   return 0;
}
